/*
 * Decompiled with CFR 0.152.
 */
package com.mckoi.database;

import com.mckoi.debug.DebugLogger;
import com.mckoi.util.ByteArrayUtil;
import com.mckoi.util.Cache;
import com.mckoi.util.IntegerVector;
import com.mckoi.util.UserTerminal;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.io.SyncFailedException;
import java.util.Arrays;

public final class FixedSizeDataStore {
    private static final int MAGIC = 195935917;
    private static final int SECTOR_DATA_OFFSET = 512;
    private static final int EXTRA_SECTOR_SIZE = 5;
    private static final byte USED = 0;
    private static final byte DELETED = -128;
    private static final boolean SECTORS_CACHED = true;
    private DebugLogger debug;
    private int sector_size;
    private File data_file;
    private RandomAccessFile data_store;
    private boolean read_only;
    private long data_store_size;
    private int sector_offset;
    private byte[] sector_buffer;
    private int buffered_sector;
    private int delete_head;
    private int used_sector_count;
    private int lock_count;
    private Cache sector_cache;
    private byte[] sync_buffer = new byte[16];
    private SectorOutputStream sector_output_stream;

    public FixedSizeDataStore(File data_file, int sector_size, boolean cache_access, DebugLogger logger) {
        this.debug = logger;
        this.sector_cache = cache_access ? new Cache(64) : null;
        this.sector_size = sector_size > 0 ? sector_size + 5 : -1;
        this.data_file = data_file;
    }

    public FixedSizeDataStore(File data_file, int sector_size, DebugLogger logger) {
        this(data_file, sector_size, true, logger);
    }

    boolean locked() {
        return this.lock_count > 0;
    }

    private int sectorCount() throws IOException {
        return (int)((this.data_store_size - (long)this.sector_offset) / (long)this.sector_size);
    }

    private long seekSector(int sector) throws IOException {
        long ra_index = sector * this.sector_size;
        long seek_to = ra_index + (long)this.sector_offset;
        this.data_store.seek(seek_to);
        return seek_to;
    }

    private void readSector(int sector) throws IOException {
        if (this.buffered_sector != sector) {
            if (this.sector_cache != null) {
                Integer cacheKey = new Integer(sector);
                byte[] sbuf = (byte[])this.sector_cache.get(cacheKey);
                if (sbuf == null) {
                    this.seekSector(sector);
                    this.data_store.readFully(this.sector_buffer, 0, this.sector_size);
                    sbuf = new byte[this.sector_size];
                    System.arraycopy(this.sector_buffer, 0, sbuf, 0, this.sector_size);
                    this.sector_cache.put(cacheKey, sbuf);
                } else {
                    System.arraycopy(sbuf, 0, this.sector_buffer, 0, this.sector_size);
                }
            } else {
                this.seekSector(sector);
                this.data_store.readFully(this.sector_buffer, 0, this.sector_size);
            }
            this.buffered_sector = sector;
        }
    }

    private void setDataStoreSize(long new_size) throws IOException {
        long p = new_size - 1L;
        if (p > 0L) {
            this.data_store.seek(p);
            this.data_store.write(0);
            this.data_store_size = new_size;
        }
    }

    private void writeSector(int sector, int length) throws IOException {
        long seek_to = this.seekSector(sector);
        if (seek_to == this.data_store_size) {
            this.setDataStoreSize(seek_to + (long)this.sector_size);
            this.seekSector(sector);
        }
        if (length <= this.sector_size) {
            this.data_store.write(this.sector_buffer, 0, length);
            if (this.sector_cache != null) {
                byte[] sbuf = new byte[this.sector_size];
                System.arraycopy(this.sector_buffer, 0, sbuf, 0, length);
                this.sector_cache.put(new Integer(sector), sbuf);
            }
        } else {
            throw new IOException("length > sector_size");
        }
    }

    private void writeSector(int sector) throws IOException {
        this.writeSector(sector, this.sector_size);
    }

    private void setSectorHeader(byte status, int next_sector) throws IOException {
        this.sector_buffer[0] = status;
        this.sector_buffer[1] = (byte)(next_sector >>> 24 & 0xFF);
        this.sector_buffer[2] = (byte)(next_sector >>> 16 & 0xFF);
        this.sector_buffer[3] = (byte)(next_sector >>> 8 & 0xFF);
        this.sector_buffer[4] = (byte)(next_sector >>> 0 & 0xFF);
    }

    private int writeBufToSector(int sector, int next_sector, byte[] buf, int offset, int length) throws IOException {
        this.setSectorHeader((byte)0, next_sector);
        System.arraycopy(buf, offset, this.sector_buffer, 5, length);
        ++this.used_sector_count;
        this.synch();
        this.writeSector(sector, length + 5);
        this.buffered_sector = sector;
        return sector;
    }

    private int reclaimTopFree() throws IOException {
        int free_sector = this.delete_head;
        this.readSector(free_sector);
        int c1 = this.sector_buffer[1] & 0xFF;
        int c2 = this.sector_buffer[2] & 0xFF;
        int c3 = this.sector_buffer[3] & 0xFF;
        int c4 = this.sector_buffer[4] & 0xFF;
        this.delete_head = (c1 << 24) + (c2 << 16) + (c3 << 8) + c4;
        return free_sector;
    }

    private int findFreeSector() throws IOException {
        if (!this.locked() && this.delete_head != -1) {
            return this.reclaimTopFree();
        }
        return this.sectorCount();
    }

    private int findFreeSectorPastNext() throws IOException {
        if (!this.locked() && this.delete_head != -1) {
            return this.reclaimTopFree();
        }
        return this.sectorCount() + 1;
    }

    private int[] findFreeSectors(int count) throws IOException {
        int fs_index;
        int[] free_sectors = new int[count];
        if (!this.locked()) {
            for (fs_index = 0; fs_index < count && this.delete_head != -1; ++fs_index) {
                free_sectors[fs_index] = this.reclaimTopFree();
            }
        }
        int sec_count = this.sectorCount();
        while (fs_index < count) {
            free_sectors[fs_index] = sec_count++;
            ++fs_index;
        }
        return free_sectors;
    }

    public long totalSize() {
        return this.data_file.length();
    }

    public void writeReservedBuffer(byte[] info, int offset, int length, int res_offset) throws IOException {
        if (length + res_offset > 128) {
            throw new Error("Attempted to write > 128 bytes in reserve buffer.");
        }
        this.data_store.seek(res_offset + 64);
        this.data_store.write(info, offset, length);
    }

    public void writeReservedBuffer(byte[] info, int offset, int length) throws IOException {
        this.writeReservedBuffer(info, offset, length, 0);
    }

    public void readReservedBuffer(byte[] info, int offset, int length) throws IOException {
        if (length > 128) {
            throw new Error("Attempted to read > 128 bytes from reserve buffer.");
        }
        this.data_store.seek(64L);
        this.data_store.readFully(info, offset, length);
    }

    public void synch() throws IOException {
        if (!this.read_only) {
            ByteArrayUtil.setLong(this.delete_head, this.sync_buffer, 0);
            ByteArrayUtil.setLong(this.used_sector_count, this.sync_buffer, 8);
            this.data_store.seek(12L);
            this.data_store.write(this.sync_buffer, 0, 16);
        }
    }

    public void hardSynch() throws IOException {
        if (!this.read_only) {
            this.synch();
            try {
                this.data_store.getFD().sync();
            }
            catch (SyncFailedException syncFailedException) {
                // empty catch block
            }
        }
    }

    public boolean isReadOnly() {
        return this.read_only;
    }

    public boolean open(boolean read_only) throws IOException {
        this.read_only = read_only;
        if (!this.data_file.exists() && this.sector_size <= 0) {
            throw new IOException("Sector size not set for new file.");
        }
        String mode = read_only ? "r" : "rw";
        this.data_store = new RandomAccessFile(this.data_file, mode);
        this.data_store.seek(0L);
        if (this.data_store.length() < 512L) {
            if (read_only) {
                throw new IOException("Unable to open FixedSizeDataStore.  No header found.");
            }
            ByteArrayOutputStream bout = new ByteArrayOutputStream(512);
            DataOutputStream dout = new DataOutputStream(bout);
            dout.writeInt(195935917);
            dout.writeInt(256);
            dout.writeInt(this.sector_size);
            dout.writeLong(-1L);
            dout.writeLong(0L);
            dout.writeByte(0);
            dout.writeInt(512);
            byte[] buf = bout.toByteArray();
            dout.close();
            byte[] buf2 = new byte[512];
            System.arraycopy(buf, 0, buf2, 0, buf.length);
            for (int i = buf.length; i < 512; ++i) {
                buf2[i] = -1;
            }
            this.data_store.write(buf2);
        }
        this.data_store.seek(0L);
        this.data_store_size = this.data_store.length();
        if (this.data_store.readInt() == 195935917) {
            int version = this.data_store.readInt();
            if (version != 256) {
                throw new IOException("Unknown version.");
            }
            int ss_check = this.data_store.readInt();
            if (this.sector_size <= 0) {
                this.sector_size = ss_check;
            }
            if (ss_check == this.sector_size) {
                int pred_size;
                boolean need_repair = false;
                this.delete_head = (int)this.data_store.readLong();
                this.used_sector_count = (int)this.data_store.readLong();
                need_repair = this.data_store.readByte() != 0;
                this.sector_offset = this.data_store.readInt();
                this.sector_buffer = new byte[this.sector_size];
                this.buffered_sector = -2;
                if (!read_only) {
                    this.data_store.seek(28L);
                    this.data_store.writeByte(1);
                }
                if ((long)(pred_size = this.sectorCount() * this.sector_size + this.sector_offset) != this.data_store_size) {
                    this.debug.write(40, this, "The FixedSizeDataStore file size is incorrect.");
                    this.debug.write(40, this, "File size should be: " + pred_size + "\n" + "But it's really: " + this.data_store_size);
                    need_repair = true;
                }
                this.data_store.seek(this.sector_offset);
                if (need_repair) {
                    this.debug.write(30, this, "Store not closed cleanly.");
                }
                return need_repair;
            }
            throw new IOException("Sector size for this data store does not match.");
        }
        throw new IOException("Format invalid; MAGIC number didn't match.");
    }

    public void close() throws IOException {
        long close_size;
        this.synch();
        if (!this.read_only) {
            this.data_store.seek(28L);
            this.data_store.writeByte(0);
        }
        if ((close_size = this.data_store.length()) != this.data_store_size) {
            this.debug.write(40, this, "On closing file, data_store_size != close_size (" + this.data_store_size + " != " + close_size + ")");
        }
        try {
            this.data_store.getFD().sync();
        }
        catch (SyncFailedException syncFailedException) {
            // empty catch block
        }
        this.data_store.close();
        this.data_store = null;
        this.sector_buffer = null;
        this.buffered_sector = -2;
    }

    public boolean isClosed() {
        return this.data_store == null;
    }

    public void delete() {
        if (this.data_store != null) {
            throw new Error("Must close before FixedSizeDataStore is deleted.");
        }
        this.data_file.delete();
    }

    public boolean exists() throws IOException {
        return this.data_file.exists();
    }

    public int getSectorSize() {
        return this.sector_size - 5;
    }

    public int getSectorUseCount() {
        return this.used_sector_count;
    }

    public int rawSectorCount() throws IOException {
        return this.sectorCount();
    }

    public void lock() {
        ++this.lock_count;
    }

    public void unlock() {
        --this.lock_count;
        if (this.lock_count < 0) {
            throw new Error("Unlocked more times than we locked.");
        }
    }

    public boolean isSectorDeleted(int sector) throws IOException {
        this.readSector(sector);
        return (this.sector_buffer[0] & 0xFFFFFF80) != 0;
    }

    public byte[] getSector(int sector, byte[] buf, int offset, int length) throws IOException {
        if (sector >= this.sectorCount()) {
            throw new IOException("Can't get sector, out of range.");
        }
        int ssize = this.getSectorSize();
        if (length > ssize) {
            throw new IOException("length > sector size");
        }
        this.readSector(sector);
        System.arraycopy(this.sector_buffer, 5, buf, offset, length);
        return buf;
    }

    public byte[] getSector(int sector, byte[] buf) throws IOException {
        return this.getSector(sector, buf, 0, Math.min(buf.length, this.getSectorSize()));
    }

    public byte[] getSector(int sector) throws IOException {
        if (sector >= this.sectorCount()) {
            throw new IOException("Can't get sector, out of range.");
        }
        int ssize = this.getSectorSize();
        byte[] buf = new byte[ssize];
        this.readSector(sector);
        System.arraycopy(this.sector_buffer, 5, buf, 0, ssize);
        return buf;
    }

    public int[] getSectorAsIntArray(int sector, int[] buf) throws IOException {
        if (sector >= this.sectorCount()) {
            throw new IOException("Can't get sector, out of range.");
        }
        int length = buf.length * 4;
        int ssize = this.getSectorSize();
        if (length > ssize) {
            throw new IOException("length > sector size");
        }
        this.readSector(sector);
        int p = 5;
        int i = 0;
        while (i < buf.length) {
            int c1 = this.sector_buffer[p++] & 0xFF;
            int c2 = this.sector_buffer[p++] & 0xFF;
            int c3 = this.sector_buffer[p++] & 0xFF;
            int c4 = this.sector_buffer[p++] & 0xFF;
            int v = (c1 << 24) + (c2 << 16) + (c3 << 8) + c4;
            buf[i++] = v;
        }
        return buf;
    }

    public int readAcross(int sector_head, byte[] buf, int offset, int length) throws IOException {
        int amount_read;
        if (sector_head >= this.sectorCount()) {
            throw new IOException("Can't get sector, out of range.");
        }
        int ssize = this.getSectorSize();
        int walk = sector_head;
        for (int to_read = length; walk != -1 && to_read > 0; to_read -= amount_read) {
            this.readSector(walk);
            if ((this.sector_buffer[0] & 0xFFFFFF80) != 0) {
                throw new IOException("Can not read across a deleted chain.");
            }
            int next_walk = ByteArrayUtil.getInt(this.sector_buffer, 1);
            amount_read = Math.min(to_read, ssize);
            System.arraycopy(this.sector_buffer, 5, buf, offset, amount_read);
            offset += amount_read;
            walk = next_walk;
        }
        return offset;
    }

    public int[] getSectorChain(int sector_head, int length) throws IOException {
        if (sector_head >= this.sectorCount()) {
            throw new IOException("Can't get sector, out of range.");
        }
        int span_count = this.calculateSectorSpan(length);
        int[] spans = new int[span_count];
        int ssize = this.getSectorSize();
        int walk = sector_head;
        for (int chain_count = 0; chain_count < span_count; ++chain_count) {
            spans[chain_count] = walk;
            this.readSector(walk);
            walk = ByteArrayUtil.getInt(this.sector_buffer, 1);
        }
        return spans;
    }

    public int[] getSectorChain(int sector_head) throws IOException {
        if (sector_head >= this.sectorCount()) {
            throw new IOException("Can't get sector, out of range.");
        }
        IntegerVector spans = new IntegerVector();
        int ssize = this.getSectorSize();
        int walk = sector_head;
        while (walk > -1) {
            spans.addInt(walk);
            this.readSector(walk);
            walk = ByteArrayUtil.getInt(this.sector_buffer, 1);
        }
        return spans.toIntArray();
    }

    public void deleteSector(int sector) throws IOException {
        this.deleteAcross(sector);
    }

    public void deleteAcross(int sector_head) throws IOException {
        if (sector_head < 0) {
            throw new IOException("Sector is out of range.");
        }
        if (sector_head >= this.sectorCount()) {
            throw new IOException("Can't get sector, out of range.");
        }
        int walk = sector_head;
        while (walk != -1) {
            this.readSector(walk);
            if ((this.sector_buffer[0] & 0xFFFFFF80) != 0) {
                throw new IOException("Sector has already been deleted.");
            }
            int next_walk = ByteArrayUtil.getInt(this.sector_buffer, 1);
            this.sector_buffer[0] = -128;
            if (next_walk == -1) {
                ByteArrayUtil.setInt(this.delete_head, this.sector_buffer, 1);
            }
            this.seekSector(walk);
            this.data_store.write(this.sector_buffer, 0, 5);
            if (this.sector_cache != null) {
                this.sector_cache.remove(new Integer(walk));
            }
            --this.used_sector_count;
            walk = next_walk;
        }
        this.delete_head = sector_head;
        this.synch();
    }

    public void deleteAllSectors() throws IOException {
        int sector_count = this.sectorCount();
        for (int i = 0; i < sector_count; ++i) {
            this.readSector(i);
            this.sector_buffer[0] = -128;
            int next = i + 1;
            if (i == sector_count - 1) {
                next = -1;
            }
            ByteArrayUtil.setInt(next, this.sector_buffer, 1);
            this.writeSector(i);
        }
        this.delete_head = sector_count == 0 ? -1 : 0;
        this.used_sector_count = 0;
        this.synch();
    }

    public int overwriteSector(int sector, byte[] buf, int offset, int length) throws IOException {
        int ssize = this.getSectorSize();
        if (length > ssize) {
            throw new IOException("Sector too large to add to store.");
        }
        return this.writeBufToSector(sector, -1, buf, offset, length);
    }

    public int overwriteSector(int sector, byte[] buf) throws IOException {
        return this.overwriteSector(sector, buf, 0, buf.length);
    }

    public int addSector(byte[] buf, int offset, int length) throws IOException {
        int ssize = this.getSectorSize();
        if (length > ssize) {
            throw new IOException("Sector too large to add to store.");
        }
        int sector = this.findFreeSector();
        return this.writeBufToSector(sector, -1, buf, offset, length);
    }

    public int addSector(byte[] buf) throws IOException {
        return this.addSector(buf, 0, buf.length);
    }

    public int calculateSectorSpan(int length) {
        int sector_size = this.getSectorSize();
        int span_count = length / sector_size;
        if (length == 0 || length % sector_size != 0) {
            ++span_count;
        }
        return span_count;
    }

    public int writeAcross(byte[] buf, int offset, int length) throws IOException {
        int sector_size = this.getSectorSize();
        int span_count = this.calculateSectorSpan(length);
        int[] free_sectors = this.findFreeSectors(span_count);
        Arrays.sort(free_sectors, 0, span_count);
        int to_write = length;
        int to_offset = 0;
        for (int i = 0; i < span_count; ++i) {
            int sector = free_sectors[i];
            int next_sector = i < span_count - 1 ? free_sectors[i + 1] : -1;
            this.writeBufToSector(sector, next_sector, buf, to_offset, Math.min(to_write, sector_size));
            to_write -= sector_size;
            to_offset += sector_size;
        }
        return free_sectors[0];
    }

    public OutputStream getSectorOutputStream() throws IOException {
        this.sector_output_stream = new SectorOutputStream();
        return this.sector_output_stream;
    }

    public int getSectorOfLastOutputStream() {
        return this.sector_output_stream.first_sector;
    }

    public int getLengthOfLastOutputStream() {
        return this.sector_output_stream.count;
    }

    public void wipeLastOutputStream() {
        this.sector_output_stream = null;
    }

    public InputStream getSectorInputStream(int sector_head) throws IOException {
        return new SectorInputStream(sector_head);
    }

    public void copyTo(File path) throws IOException {
        String fname = this.data_file.getName();
        FileOutputStream fout = new FileOutputStream(new File(path, fname));
        int BUF_SIZE = 65536;
        byte[] buf = new byte[BUF_SIZE];
        this.data_store.seek(0L);
        int read = this.data_store.read(buf, 0, BUF_SIZE);
        while (read >= 0) {
            fout.write(buf, 0, read);
            read = this.data_store.read(buf, 0, BUF_SIZE);
        }
        fout.close();
    }

    public void fix(UserTerminal terminal) throws IOException {
        int i;
        terminal.println("- File: " + this.data_file);
        this.synch();
        if ((this.data_store_size - (long)this.sector_offset) % (long)this.sector_size != 0L) {
            terminal.println("+ Altering length of file so it is correct for sector size");
            int row_count = this.sectorCount() + 1;
            long new_size = row_count * this.sector_size + this.sector_offset;
            this.setDataStoreSize(new_size);
        }
        IntegerVector sector_info = new IntegerVector();
        IntegerVector scc = new IntegerVector();
        int null_count = 0;
        int sector_count = this.sectorCount();
        terminal.println("- Sector Count: " + this.sectorCount());
        for (int i2 = 0; i2 < sector_count; ++i2) {
            this.readSector(i2);
            int next_chain = ByteArrayUtil.getInt(this.sector_buffer, 1);
            sector_info.addInt(this.sector_buffer[0]);
            sector_info.addInt(next_chain);
            if (next_chain == -1) {
                ++null_count;
                continue;
            }
            int old_val = 0;
            if (next_chain < scc.size()) {
                old_val = scc.intAt(next_chain);
            }
            scc.placeIntAt(old_val + 1, next_chain);
        }
        terminal.println("- unchained sectors = " + null_count);
        IntegerVector bad_sectors = new IntegerVector();
        for (i = 0; i < scc.size(); ++i) {
            int ref_count = scc.intAt(i);
            if (ref_count <= 1) continue;
            terminal.println("- [" + i + "] reference count = " + ref_count);
            terminal.println("+ Marking all references as bad (except first).");
            boolean found_first = false;
            for (int n = 0; n < sector_info.size(); n += 2) {
                if (sector_info.intAt(n + 1) != i) continue;
                if (found_first) {
                    bad_sectors.addInt(n / 2);
                }
                found_first = true;
            }
        }
        if (bad_sectors.size() > 0) {
            terminal.println("+ Marked " + bad_sectors.size() + " sectors bad.");
        }
        for (i = 0; i < bad_sectors.size(); ++i) {
            int sector = bad_sectors.intAt(i);
            this.readSector(sector);
            this.sector_buffer[0] = -128;
            this.writeSector(sector);
        }
        this.repair();
    }

    public boolean clearDeletedSectors() throws IOException {
        if (this.locked()) {
            throw new IOException("Store is locked, can not reclaim deleted sectors.");
        }
        if (this.delete_head != -1) {
            int scount = this.sectorCount();
            int move_to = 0;
            int row_count = 0;
            for (int i = 0; i < scount; ++i) {
                this.readSector(i);
                if ((this.sector_buffer[0] & 0xFFFFFF80) != 0) continue;
                ++row_count;
                if (move_to < i) {
                    this.writeSector(move_to);
                    this.buffered_sector = move_to;
                }
                ++move_to;
            }
            long new_size = row_count * this.sector_size + this.sector_offset;
            this.setDataStoreSize(new_size);
            this.delete_head = -1;
            this.used_sector_count = row_count;
            this.synch();
            return true;
        }
        return false;
    }

    public void repair() throws IOException {
        this.delete_head = -1;
        int scount = this.sectorCount();
        int row_count = 0;
        int delete_count = 0;
        byte[] mark_buffer = new byte[5];
        for (int i = 0; i < scount; ++i) {
            this.readSector(i);
            if ((this.sector_buffer[0] & 0xFFFFFF80) != 0) {
                int v = this.delete_head;
                mark_buffer[0] = -128;
                mark_buffer[1] = (byte)(v >>> 24 & 0xFF);
                mark_buffer[2] = (byte)(v >>> 16 & 0xFF);
                mark_buffer[3] = (byte)(v >>> 8 & 0xFF);
                mark_buffer[4] = (byte)(v >>> 0 & 0xFF);
                this.seekSector(i);
                this.data_store.write(mark_buffer, 0, 5);
                if (this.sector_cache != null) {
                    this.sector_cache.remove(new Integer(i));
                }
                this.delete_head = i;
                ++delete_count;
                continue;
            }
            ++row_count;
        }
        this.used_sector_count = row_count;
        this.synch();
        this.debug.write(10000, this, "Repair found (" + delete_count + ") deleted, (" + row_count + ") used sectors.");
    }

    public String statusString() throws IOException {
        int sc = this.sectorCount();
        StringBuffer str = new StringBuffer();
        str.append("Sector Count: ");
        str.append(sc);
        str.append("\nSectors Used: ");
        str.append(this.getSectorUseCount());
        str.append("\nLocks: ");
        str.append(this.lock_count);
        str.append("\nFree Sectors: ");
        str.append(sc - this.getSectorUseCount());
        str.append("\n");
        return new String(str);
    }

    private final class SectorInputStream
    extends InputStream {
        private int sector;
        private int index;
        private int count;
        private byte[] sector_buffer;

        SectorInputStream(int sector_head) throws IOException {
            this.sector = sector_head;
            this.sector_buffer = FixedSizeDataStore.this.sector_buffer;
            this.loadNextSector();
            this.count = 0;
        }

        private void loadNextSector() throws IOException {
            if (this.sector != -1) {
                FixedSizeDataStore.this.readSector(this.sector);
            }
            this.index = 5;
            this.sector = ByteArrayUtil.getInt(this.sector_buffer, 1);
        }

        public final int read() throws IOException {
            int b = this.sector_buffer[this.index] & 0xFF;
            ++this.index;
            ++this.count;
            if (this.index >= FixedSizeDataStore.this.sector_size) {
                this.loadNextSector();
            }
            return b;
        }

        public int read(byte[] b, int offset, int len) throws IOException {
            int original_len = len;
            while (this.index + len > FixedSizeDataStore.this.sector_size) {
                int to_copy = FixedSizeDataStore.this.sector_size - this.index;
                System.arraycopy(this.sector_buffer, this.index, b, offset, to_copy);
                offset += to_copy;
                len -= to_copy;
                this.index += to_copy;
                this.count += to_copy;
                this.loadNextSector();
            }
            if (len > 0) {
                System.arraycopy(this.sector_buffer, this.index, b, offset, len);
                this.index += len;
                this.count += len;
                if (this.index >= FixedSizeDataStore.this.sector_size) {
                    this.loadNextSector();
                }
            }
            return original_len;
        }

        public long skip(long len) throws IOException {
            long original_len = len;
            while ((long)this.index + len > (long)FixedSizeDataStore.this.sector_size) {
                int to_copy = FixedSizeDataStore.this.sector_size - this.index;
                len -= (long)to_copy;
                this.index += to_copy;
                this.count += to_copy;
                this.loadNextSector();
            }
            if (len > 0L) {
                this.index = (int)((long)this.index + len);
                this.count = (int)((long)this.count + len);
                if (this.index >= FixedSizeDataStore.this.sector_size) {
                    this.loadNextSector();
                }
            }
            return original_len;
        }
    }

    private class SectorOutputStream
    extends OutputStream {
        private final byte[] buf;
        private int first_sector = -1;
        private int cur_sector = -1;
        private int last_sector = -1;
        private int index;
        private int count;

        SectorOutputStream() throws IOException {
            this.buf = new byte[FixedSizeDataStore.this.getSectorSize()];
            this.index = 0;
            this.count = 0;
            this.cur_sector = this.first_sector = FixedSizeDataStore.this.findFreeSector();
        }

        public void write(int b) throws IOException {
            if (this.index >= this.buf.length) {
                int next_sector = FixedSizeDataStore.this.findFreeSector();
                if (next_sector == this.cur_sector) {
                    ++next_sector;
                }
                FixedSizeDataStore.this.writeBufToSector(this.cur_sector, next_sector, this.buf, 0, this.index);
                this.cur_sector = next_sector;
                this.index = 0;
            }
            this.buf[this.index] = (byte)b;
            ++this.index;
            ++this.count;
        }

        public void write(byte[] b, int offset, int len) throws IOException {
            while (this.index + len > this.buf.length) {
                int to_copy = this.buf.length - this.index;
                System.arraycopy(b, offset, this.buf, this.index, to_copy);
                offset += to_copy;
                len -= to_copy;
                this.index += to_copy;
                this.count += to_copy;
                int next_sector = FixedSizeDataStore.this.findFreeSector();
                if (next_sector == this.cur_sector) {
                    ++next_sector;
                }
                FixedSizeDataStore.this.writeBufToSector(this.cur_sector, next_sector, this.buf, 0, this.index);
                this.cur_sector = next_sector;
                this.index = 0;
            }
            if (len > 0) {
                System.arraycopy(b, offset, this.buf, this.index, len);
                this.index += len;
                this.count += len;
            }
        }

        public void flush() throws IOException {
        }

        public void close() throws IOException {
            FixedSizeDataStore.this.writeBufToSector(this.cur_sector, -1, this.buf, 0, this.index);
        }
    }
}

